package net.yaranga.framework

import io.r2dbc.postgresql.PostgresqlConnectionFactoryProvider.OPTIONS
import io.r2dbc.spi.Connection
import io.r2dbc.spi.ConnectionFactories
import io.r2dbc.spi.ConnectionFactoryOptions.*
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.test.StandardTestDispatcher
import kotlinx.coroutines.test.runTest
import kotlinx.coroutines.test.setMain
import org.jetbrains.exposed.v1.core.dao.id.LongIdTable
import org.jetbrains.exposed.v1.r2dbc.R2dbcDatabase
import org.jetbrains.exposed.v1.r2dbc.SchemaUtils
import org.jetbrains.exposed.v1.r2dbc.insert
import org.jetbrains.exposed.v1.r2dbc.transactions.suspendTransaction
import org.junit.jupiter.api.AfterEach
import org.junit.jupiter.api.BeforeEach
import org.junit.jupiter.api.Test
import org.junit.jupiter.api.TestInfo
import org.testcontainers.containers.PostgreSQLContainer
import org.testcontainers.images.PullPolicy
import org.testcontainers.junit.jupiter.Container
import org.testcontainers.junit.jupiter.Testcontainers
import org.testcontainers.shaded.org.apache.commons.lang3.RandomStringUtils
import reactor.core.publisher.Mono
import java.util.*


@Testcontainers
class TransactionIOTest() {
    companion object {
        @Container
        val postgresContainer = PostgreSQLContainer("postgres:17-alpine")
            .withImagePullPolicy(PullPolicy.alwaysPull())
            .withReuse(true)
            .withUrlParam("TC_REUSABLE", "true")
    }

    private lateinit var testDBName: String
    lateinit var connection: Mono<Connection>

    @OptIn(ExperimentalCoroutinesApi::class)
    @BeforeEach
    fun setUp(testInfo: TestInfo) {
        Dispatchers.setMain(StandardTestDispatcher())

        testDBName = testInfo.testClass.get().simpleName +
                "-" +
                testInfo.testMethod.get().name +
                "-" +
                RandomStringUtils.randomAlphanumeric(6).lowercase(Locale.ENGLISH)

        val options: MutableMap<String?, String?> = HashMap<String?, String?>()
        options.put("lock_timeout", "10s")


        val connectionFactory = ConnectionFactories.get(
            builder()
                .option(DRIVER, "postgresql")
                .option(HOST, postgresContainer.host)
                .option(PORT, postgresContainer.getMappedPort(PostgreSQLContainer.POSTGRESQL_PORT))
                .option(USER, postgresContainer.username)
                .option(PASSWORD, postgresContainer.password)
                .option(DATABASE, postgresContainer.databaseName) // optional
                .option(OPTIONS, options) // optional
                .build()
        )

        connection = Mono.from(connectionFactory.create())

        connection.flatMapMany { c ->
            c.createStatement("CREATE DATABASE \"${testDBName}\" ENCODING 'utf8'")
                .execute()
        }.blockLast() // Block until the database creation is complete
    }

    @AfterEach
    fun tearDown() {
        connection.flatMapMany { c -> c.close() }.blockLast() // Block until connection is closed
    }

    object UsersTable : LongIdTable("users") {
        val name = varchar("name", 50)
    }

    @Test
    fun testNested() = runTest {
        val db = createDB()

        suspendTransaction(db = db) {
            SchemaUtils.create(UsersTable)
        }

        suspendTransaction(db = db) {
            UsersTable.insert {
                it[name] = "Alice"
            }
        }
    }

    private fun createDB(): R2dbcDatabase {
        val options: MutableMap<String?, String?> = HashMap<String?, String?>()
        options.put("lock_timeout", "10s")

        return R2dbcDatabase.connect {
            useNestedTransactions = true
            defaultMaxAttempts = 1

            connectionFactoryOptions {
                option(DRIVER, "postgresql")
                option(HOST, postgresContainer.host)
                option(PORT, postgresContainer.getMappedPort(PostgreSQLContainer.POSTGRESQL_PORT))
                option(USER, postgresContainer.username)
                option(PASSWORD, postgresContainer.password)
                option(DATABASE, testDBName)
                option(OPTIONS, options)
            }
        }
    }
}
