import os
import json
import random
import string
from copy import deepcopy


# Function to read templates
def read_templates():
    insight_template = json.loads(open('insight_template.json','r').read())
    measure_template = json.loads(open('measure_template.json','r').read())
    attribute_template = json.loads(open('attribute_template.json','r').read())
    return (insight_template, measure_template, attribute_template)

# Helper function to add measure/metric to insight meta data
def add_measure(insight_template, measure_template, measures_meta, 
                measure_localIdentifier='measures'):
    measure_dict = {'items': []}
    # Going over each fact/metric and add into bucket
    for measure in measures_meta:
        curr_measure = deepcopy(measure_template)
        # Create random string local identifier
        measure_id = ''.join(random.choice(string.ascii_uppercase + string.digits) for _ in range(16))
        # Fill in a measure metadata
        curr_measure['measure']['localIdentifier'] = measure_id
        curr_measure['measure']['title'] = measure['measure_title']
        curr_measure['measure']['definition']['measureDefinition']['item']['identifier']['id'] = measure['measure_id']
        curr_measure['measure']['definition']['measureDefinition']['item']['identifier']['type'] = measure['measure_type']
        # If it is a fact, it requires to put the type of aggregation
        if measure['measure_type']=='fact':
            if 'measure_aggregation' in measure:
                curr_measure['measure']['definition']['measureDefinition']['aggregation'] = measure['measure_aggregation']
            else:
                curr_measure['measure']['definition']['measureDefinition']['aggregation'] = 'sum'
        measure_dict['items'].append(curr_measure)
    # Put localIdentifier for measure metadata
    measure_dict['localIdentifier'] = measure_localIdentifier
    insight_template['data']['attributes']['content']['buckets'].append(measure_dict)
    return insight_template

# Helper function to add attribute to insight meta data
def add_attribute(insight_template, attribute_template, attributes_meta,
                  attribute_localIdentifier='attribute'):
    attribute_dict = {'items': []}
    for attribute in attributes_meta:
        print(attribute)
        curr_attribute = deepcopy(attribute_template)
        col_localIdentifier = ''.join(random.choice(string.ascii_uppercase + string.digits) for _ in range(16))
        curr_attribute['attribute']['localIdentifier'] = col_localIdentifier
        curr_attribute['attribute']['displayForm']['identifier']['id'] = attribute['attribute_id']
        curr_attribute['attribute']['displayForm']['identifier']['type'] = attribute['attribute_type']
        attribute_dict['items'].append(curr_attribute)
    # Put localIdentifier for attribute metadata, possible value: attribute, trend, segment, view
    attribute_dict['localIdentifier'] = attribute_localIdentifier
    insight_template['data']['attributes']['content']['buckets'].append(attribute_dict)
    return insight_template

# Helper function to find local identifier of attribute if exist
def find_sorting_identifier(insight_template, filter_id):
    identifier = None
    for item in insight_template['data']['attributes']['content']['buckets']:
        # As long as it is attribute, always pick the first element
        if 'items' in item and 'attribute' in item['items'][0]:
            identifier = item['items'][0]['attribute']['localIdentifier']
    return identifier

# Helper function to add sorting meta data
def add_sort(insight_template, attribute_id, direction='asc'):
    sort_val = {"attributeSortItem": {
                        "attributeIdentifier": attribute_id,
                        "direction": direction}}
    insight_template['data']['attributes']['content']['sorts'].append(sort_val)
    return insight_template

# Helper function to add filter to insight (Table)
def add_filter(insight_template, meta):
    # Filter by attribute value
    if 'positiveAttributeFilter' in meta:
        vals = meta['positiveAttributeFilter']
        # Fill out metadata
        filter_val = {"positiveAttributeFilter": {
                        "displayForm": {
                              "identifier": {
                                    "id": vals['filter_id'],
                                 "type": vals['filter_type']}},
                          "in": {"values": vals['filter_values']}}}
        insight_template['data']['attributes']['content']['filters'].append(filter_val)
    # Filter for Top/Bottom n
    elif 'rankingFilter' in meta:
        vals = meta['rankingFilter']
        attribute_id = ''.join(random.choice(string.ascii_uppercase + string.digits) for _ in range(16))
        # Obtain the identifier from insight_template
        identifier = None
        # Find the localIdentifier of targeted measures
        for item in insight_template['data']['attributes']['content']['buckets']:
            if 'items' in item and 'measure' in item['items'][0] and \
                item['items'][0]['measure']['definition']['measureDefinition']['item']['identifier']['id'] == vals['filter_id']:
                    identifier = item['items'][0]['measure']['localIdentifier']
        # Fill out metadata
        filter_val = {"rankingFilter": {
                        "measure": {
                          "localIdentifier": identifier},
                      "operator": vals['operator'],
                      "value": vals['value']}}
        insight_template['data']['attributes']['content']['filters'].append(filter_val)
        insight_template = add_sort(insight_template, attribute_id, 'asc')
    return insight_template


"""
Fill out the new insight meta data
The expected structures:
measures_meta = [{}, {}]
    Each dictionary:{'measure_title': str - Name to display on AD,
                     'measure_id': str - object_id for fact or metric,
                     'measure_type': str - measure or metic,
                     'measure_aggregation': str (Optional, only if there is fact)
                                              - sum, avg, min, max...}
attribute_meta = [{}, {}]
    Each dictionary:{'attribute_id': str - object_id for attribute,
                     'attribute_type': str - label}

filter_meta = {}
Two options:
    Filter by attribute value
    Dictionary:{'positiveAttributeFilter':{'filter_id': str - object_id for attribute
                                           'filter_type': str - label
                                           'filter_value':[] - attribute value to be filtered}}
    Filter by Top/Bottom n
    Dictionary:{'rankingFilter':{'filter_id': str - object_id for fact or metric
                                 'operator': str - Fill in Top or Bottom
                                 'value': int - number of top or bottom n}}
"""
def create_insight_meta(insight_title:str, insight_id:str, 
                        insight_description:str, insight_template:dict, 
                        measure_template:dict, attribute_template:dict,
                        measures_meta:list, attributes_meta:list, 
                        viz_type:str, filter_meta:dict = None):
    # Fill out AD title, ID, and description
    insight_template['data']['attributes']['title'] = insight_title
    insight_template['data']['id'] = insight_id
    insight_template['data']['attributes']['description'] = insight_description

    # Now add measures
    insight_template = add_measure(insight_template, measure_template, measures_meta)

    # Set visualization type
    if viz_type == 'line':
        insight_template = add_attribute(insight_template, attribute_template, attributes_meta, 'trend')
        insight_template['data']['attributes']['content']['visualizationUrl'] = 'local:line'
    # Headline need not attribute at all
    elif viz_type == 'headline':
        insight_template['data']['attributes']['content']['visualizationUrl'] = 'local:headline'
    # Bar chart may use default setting in attribute_template
    elif viz_type == 'bar' or viz_type == 'column':
        insight_template = add_attribute(insight_template, attribute_template, attributes_meta, 'view')
        insight_template['data']['attributes']['content']['visualizationUrl'] = 'local:column'
    elif viz_type == 'table':
        insight_template = add_attribute(insight_template, attribute_template, attributes_meta)
        insight_template['data']['attributes']['content']['visualizationUrl'] = 'local:table'
        # Randomly pick the first attribute as the attribute for sorting
        sorting_attribute = find_sorting_identifier(insight_template, attributes_meta[0]['attribute_id'])
        insight_template = add_sort(insight_template, sorting_attribute, 'asc')
    elif viz_type == 'pie':
        insight_template = add_attribute(insight_template, attribute_template, attributes_meta, 'view')
        insight_template['data']['attributes']['content']['visualizationUrl'] = 'local:pie'
        insight_template['data']['attributes']['content']['properties'] = {'controls':{'dataLabels':{'visible':True}}}
    elif viz_type == 'donut':
        insight_template = add_attribute(insight_template, attribute_template, attributes_meta, 'view')
        insight_template['data']['attributes']['content']['visualizationUrl'] = 'local:donut'
        insight_template['data']['attributes']['content']['properties'] = {'controls':{'dataLabels':{'visible':True}}}
    else:
        print('Currently not support yet!')
        return None
    
    # Now handle the filters
    if filter_meta is not None:
        # Add filter to template
        insight_template = add_filter(insight_template, filter_meta)
    return insight_template

# Call API to execute the AD creation
def execute_insight_creation(host, workspace_id, metadata):
    cmd = f'''curl -H "Authorization: Bearer YWRtaW46Ym9vdHN0cmFwOmFkbWluMTIz" \
             -H "Content-Type: application/vnd.gooddata.api+json" \
             -H "Accept: application/vnd.gooddata.api+json" \
             -X POST '''
    cmd += f'{host}/api/entities/workspaces/{workspace_id}/visualizationObjects'
    cmd += f' -d \''
    cmd += json.dumps(metadata)
    cmd += '\''
    os.system(cmd)

# Call API to delete insight, for testing purpose
def delete_insight(host, workspace_id, insight_id):
    cmd = '''curl -H "Authorization: Bearer YWRtaW46Ym9vdHN0cmFwOmFkbWluMTIz" \
                -H "Content-Type: application/vnd.gooddata.api+json" \
                -H "Accept: application/vnd.gooddata.api+json" \
                -X DELETE '''
    cmd += f'{host}/api/entities/workspaces/{workspace_id}/visualizationObjects/{insight_id}'
    os.system(cmd)

